/*
** text.c
** NTSC/PAL Video Generator using: 
**      T3      Interrupt for main timing
**      OC3     Horizontal Synchronization pulse
**      OC4     SPI buffer reload timing
**      SPI1    pixel serialization (8x16-bit FIFO)
**
** Author: Lucio Di Jasio mailto:pilot@flyingpic24.com
**     rev 2.0 - 06/06/06
**     rev 2.1 - 01/07/08 disabled sck
**     rev 3.0 - 04/28/08 OC3 in PWM mode hw optimization
** Pin assignements:                  
**          SDO1 - RF8 - VIDEO
**          OC3  - RD2 - SYNC (requires AV16/32 board)
*/
#include "EX16.h"
#include "text.h"
//#include "timer_1ms.h"
#include "font.h"
#include <pps.h>
#include <stdio.h>
//#include "timer_1ms.h"

//#include "printf.h"

#define VRES (ROWS*8)
#define HRES (COLS*8)


// timing definitions for video vertical state machine
#ifdef PAL
    #define V_LINES  312    // total number of lines in frame
    #define LINE_T  1024    // total number of Tcy in a line
#else  // default NTSC configuration
    #define V_LINES  262    // total number of lines in frame
    #define LINE_T  1016    // total number of Tcy in a line
#endif

#define VSYNC_N     3      // V sync lines

// count the number of remaining black lines top+bottom
#define VBLANK_N    (V_LINES -VRES - VSYNC_N)  

#define PREEQ_N   VBLANK_N /2        // pre equ + bottom blank 
#define POSTEQ_N  VBLANK_N - PREEQ_N // post eq + top blank 

// definition of the vertical sync state machine
#define SV_PREEQ    0
#define SV_SYNC     1
#define SV_POSTEQ   2
#define SV_LINE     3

// timing definitions for video horizontal state machine
#define HSYNC_T  85     // Tcy in a horizontal synch pulse
#define BPORCH_T 90     // Tcy in a back porch
// Tcy in each pixel, valid values are only 2 or 3
#define PIX_T    3      // Aspect Ratio 2 = 16:9, 3 = 4:3
// jason jason, for the keyboar only, you use 3 to get a full screen
// with sd card, it will be necessary to use th value of 2

// use 2 (16:9) to allow for up to 48 columns of tex
// Text Page array
char  VMap[ COLS * ROWS];

 char KBuff [256];
char *VPtr, *LPtr;

// reordered Font
unsigned char RFont[ F_SIZE*8];
unsigned char *FPtr;

int HCount, VCount, RCount, VState; 

// next state table
int VS[4] = { SV_SYNC, SV_POSTEQ, SV_LINE, SV_PREEQ};
// next counter table
int VC[4] = { VSYNC_N,  POSTEQ_N,    VRES,  PREEQ_N};

int blink;
int cursor_on;






void _ISRFAST _T2Interrupt( void)
{
    _RA0=1;
    // advance the state machine
    if ( --VCount == 0)
    {
        VCount = VC[ VState];
        VState = VS[ VState];
     }
    // vertical state machine
    switch ( VState) {
        case SV_PREEQ:  // 0
                LPtr = VMap;
                RCount = 0;
            break;
            
        case SV_SYNC:   // 1
            // vertical sync pulse
            OC3R = LINE_T - HSYNC_T - BPORCH_T;
            break;
            
        case SV_POSTEQ: // 2
            // horizontal sync pulse
            OC3R = HSYNC_T;
            break;
            
        default:            
        case SV_LINE:   // 3
            // activate OC4 for the SPI loading
            OC4R = HSYNC_T + BPORCH_T;
     OC4CON1bits.OCTSEL = 0;     // Clock source is Timer 3
     OC4CON1bits.OCM = 0b001;           // Single shot mode, oc4
     
     OC4CON1bits.TRIGMODE = 0; 
    // OC4CON2 = 0;
     OC4CON2bits.SYNCSEL = 0b1100; // Sync to Timer 1 
     OC4CON2bits.OCTRIG = 0;
            
            HCount = HRES/128;  // reload counte
            // prepare the font pointer
            FPtr = &RFont[ RCount * F_SIZE];
            // prepare the line pointer
            VPtr = LPtr;
                                               
            // Advance the RCount
            if ( ++RCount == 8)
            { 
                RCount = 0;
                LPtr += COLS;
            }
    } //switch

    // clear the interrupt flag
    _T2IF = 0;
    _RA0=0;

} // T3Interrupt




void _ISRFAST _OC4Interrupt( void) 
{    
_RA1=1;
    unsigned i,k;
    // load SPI buffers indirectly from font 
    i=8;
    do{
        k = FPtr[ *VPtr++];
        SPI1BUF = ( k << 8) | FPtr[ *VPtr++];;
    } while( --i >0);

    if ( --HCount > 0)
    { // activate again in time for the next SPI load
            OC4R += ( PIX_T * 8 * 16); // 8 * 16 originally
//                        OC4R += ( 296); // 8 * 16 originally
     OC4CON1bits.OCTSEL = 0;     // Clock source is Timer 3
     OC4CON1bits.OCM = 0b001;           // Single shot mode, oc4
     
     OC4CON1bits.TRIGMODE = 0; 
    // OC4CON2 = 0;
     OC4CON2bits.SYNCSEL = 0b1100; // Sync to Timer 1 
     OC4CON2bits.OCTRIG = 0;         
    }  
    // clear the interrupt flag
    _OC4IF = 0;
_RA1=0;

} // OC4Interrupt







void InitVideo( void)
{
    int i, j;
    const char *p;
    unsigned char *r = RFont;
    
    // 1. prepare a reversed font table
    for (i=0; i<8; i++)
    {
        p = Font8x8 + i;
        for (j=0; j<F_SIZE; j++)
        {
            *r++ = *p;
            p += 8;
        } // for j
    }  // for i   
         
    // 2. set the priority levels
    _T2IP = 7;      // jsonjason i swtiched these to 7 and it got rid of 
    _OC4IP = 7;

    TMR2 = 0;       // clear the timer
    PR2 = LINE_T;   // set the period register to NTSC line 
    
    // 3. configure Timer3 modules
    T2CON = 0x8000; // enabled, prescaler 1:1, internal clock 
 
    // 4. configure OC3 in continuous pulse mode
    OC3R = HSYNC_T;
    OC3RS = 0;
    
    //OC3CON = 0x000d;    // this is the 128 code -- - -- ATTENTION, JASON JASON JASON -- which is 1101
// single pulse, with timer 3
   //  OC3CON1 = 0;
     OC3CON1bits.OCTSEL = 0;     // Clock source is Timer 3
     OC3CON1bits.OCM = 0b111;           // Single compare, Continuous Pulse mode
 
 //    OC3CON2 = 0;
    OC3CON2bits.SYNCSEL = 0b1100; // Sync to Timer 3
    OC3CON2bits.OCTRIG = 0;
    OC3CON1bits.TRIGMODE = 0; 

    // 5. init Timer3/OC4 Interrupts, clear the flags
    _OC4IF = 0; _OC4IE = 1;
    _T2IF = 0;  _T2IE = 1;
        
    // 6. init the SPI1
    // Master, 16 bit, disable SCK, disable SS

        SPI1CON1 = 0x1437;  // prescaler 1:3

    
    SPI1CON2 = 0x0001;      // Enhanced mode, 8x FIFO
    SPI1STAT = 0x8000;      // enable SPI port
            
    // 7. init the vertical sync state machine
    VState = SV_PREEQ;
    VCount = PREEQ_N;
    
} // InitVideo


void ClearScreen( void)
{
    int i, j;
    char *v;
    
    v = VMap;
    
    // clear the screen     
    for ( i=0; i < (ROWS); i++)
        for ( j=0; j < (COLS); j++)
           *v++ = ' ' - F_OFFS;
} // ClearScreen


void HaltVideo( void)
{
    T2CONbits.TON = 0;   // turn off the vertical state machine
} // HaltVideo




int cx, cy;

void putcV(int a) // use putcV for collecting keyboard inputs
{    
    
    // check if char in font range
    a -= F_OFFS; // this relates to description of the font, set it up by taking 20 off the asci value
// that is, a is equal to the Font, minus the offset (20)
   // i do not see how a is related to the array
    if ( a < 0)         a = 0;
    if ( a >= F_SIZE)   a = F_SIZE-1; // this is to do with font seclection
    // check page boundaries 
    if ( cx >= COLS-1)        // wrap around x
    {
        cx = 0;
        cy++;   
    } 
    cy %= ROWS;             // wrap around y, // when cy reaches the rows number, make it one again
    // find first row in the video map
    VMap[ cy * COLS + cx] = a; // this is where the font is aligned to the map
    // increment cursor position
    cx++; // this is why there is indentation
} // putcV


unsigned char old_rows;
unsigned char new_rows;
unsigned char characters = 0;
unsigned char max_columns = 32;
unsigned char rows = 0;
unsigned char do_not_scroll;
long length = 10000;
char prev;
int end_file; 

void putsV1( char *s, unsigned char max_rows, unsigned char min_rows) // note! the putsV function behaves very like the print c functions. Use \n for new line, etc

{      
    while (length --)
    {
        while(rows < min_rows || rows > max_rows) // DO NOT PRINT if the document is out of row view
        {
        switch (*s)
        {
            case 0x00:  //end of file
                return;
                
            case 0x0D: // carraige ret
                //rows++;
                //characters = 0;

            case 0x0A:  // new line
                rows++;
                characters = 0;
                
            case 0x20: // space
                // ? how to roll it over? 
                
            default:    // character
                characters ++;
                *s++;
                if(characters == max_columns)
                {
                    rows++;
                    characters = 0;
                }
             //   end_file = 10; // set 
                       break; 
        }
    }
       
        while(rows < max_rows && rows >= min_rows) // PRINT NOW, IN RANGE
        {
        switch (*s)
        {
            case 0x1A :  //end of file
              //  end_file = rows;
                return;
                
            case 0x0D: // carraige ret
              //  rows++;
             //   characters = 0;
             //   cy++;
             //   cx = 0;

            case 0x0A:  // new line
                rows++;
                characters = 0;
                cy++;
                cx = 0;
                
            case 0x20: // space
                // ? how to roll it over? 
                
            default:    // character
                characters ++;
                
                if(characters == max_columns-1)
                {
                
                    characters = 0;
                    cx = 0;
                    cy ++;
                    rows++;                         
                    cy ++;
                    rows++;                    
                }
                putcV(*s++);
                
                new_rows = rows; 
                if(new_rows > old_rows)
                {
                old_rows = new_rows;
                end_file = old_rows;
                
                }
                       break; 
        }
 
    }
                       if(rows >= max_rows)
                {
                           rows = 0;
                           characters = 0;
                       break; 
                } 

    } 
}


                

void putsV( char *s)
{
    while (*s)
        putcV(*s++);
}


void pcr( void)
{
    cx = 0; 
    cy++;
    cy %= ROWS;
} // pcr





